// server.js (updated)
const { Client, LocalAuth, MessageMedia } = require('whatsapp-web.js');
const qrcode = require('qrcode-terminal');
const express = require('express');
const cors = require('cors');
const multer = require('multer');
const fs = require('fs');
const path = require('path');

const app = express();
app.use(cors());
app.use(express.json());


// ====================== KONFIGURASI ======================
const PORT = 470;
const myToken = 'r4h4s14';

// ====================== OUTBOX EXECUTOR ======================
const OUTBOX_URL = 'http://localhost:8000/process-outbox-direct?token=rahasia';
const MIN_DELAY = 10 * 1000;   // 10 detik
const MAX_DELAY = 120 * 1000;  // 2 menit

// ====================== STATUS GLOBAL ======================
let serverStatus = 'init';   // init | qrcode | waReady
let qrCode = '';

// ====================== INIT WHATSAPP ======================
const client = new Client({
    puppeteer: { headless: true },
    authStrategy: new LocalAuth(),
    webVersionCache: { type: 'none' }
});

// ====================== EVENT WHATSAPP ======================
client.on('qr', (qr) => {
    serverStatus = 'qrcode';
    qrCode = qr;
    qrcode.generate(qr, { small: true });
    console.log('QR Code tersedia');
});

client.on('ready', () => {
    serverStatus = 'waReady';
    qrCode = '';
    console.log('Client Siap !');
});

client.on('authenticated', () => {
    console.log('WhatsApp Terautentikasi');
});

client.initialize();

// ====================== MULTER (upload) ======================
const uploadDir = path.join(__dirname, 'uploads');
if (!fs.existsSync(uploadDir)) fs.mkdirSync(uploadDir);

const storage = multer.diskStorage({
    destination: (req, file, cb) => cb(null, uploadDir),
    filename: (req, file, cb) => cb(null, `${Date.now()}-${file.originalname}`)
});
const upload = multer({ storage });

// ====================== UTIL ======================
function formatNumberToWhatsAppId(number) {
    if (!number) return null;
    // hapus semua non-digit
    let digits = number.toString().replace(/\D/g, '');
    // Jika sudah memakai suffix (mis. contains @), kembalikan apa adanya
    if (number.includes('@')) return number;
    // Jangan otomatis ubah leading 0 -> country code,
    // user diharapkan memasukkan nomor internasional (contoh: 628123456789)
    return `${digits}@c.us`;
}

// ====================== API ENDPOINT ======================

// 1️⃣ CEK STATUS + QR CODE
app.get('/', (req, res) => {
    if (req.query.token !== myToken) {
        return res.status(401).json({ error: 'Token Invalid' });
    }

    res.json({
        serverstatus: serverStatus,
        qrcode: qrCode
    });
});

// 2️⃣ KIRIM PESAN TEKS
// POST /send_message  Content-Type: application/json
// body: { token, number, message }
app.post('/send_message', async (req, res) => {
    const { token, number, message } = req.body;
    if (token !== myToken) {
        return res.status(401).json({ error: 'Token Invalid' });
    }
    if (!number || !message) {
        return res.status(400).json({ error: 'number and message are required' });
    }
    if (serverStatus !== 'waReady') {
        return res.status(400).json({ error: 'WhatsApp belum siap' });
    }

    try {
        const chatId = formatNumberToWhatsAppId(number);
        const sent = await client.sendMessage(chatId, message, {sendSeen: false });
        return res.json({ status: 'ok', id: sent.id._serialized });
    } catch (err) {
        console.error(err);
        return res.status(500).json({ error: err.message });
    }
});

// 3️⃣ KIRIM FILE (img, pdf, doc, png, etc)
// POST /send_file  multipart/form-data
// fields: token, number, caption (opsional), asDocument (opsional: 'true'/'false')
// file field name: 'file'
// Cari bagian sendFile endpoint, dan perbaiki parameter caption:
app.post('/send_file', upload.single('file'), async (req, res) => {
    const token = req.body.token || req.query.token;
    const number = req.body.number || req.query.number;
    const caption = req.body.caption || '';
    const asDocument = (req.body.asDocument === 'true') || (req.body.asDocument === true);

    if (token !== myToken) {
        // hapus file sementara jika ada
        if (req.file && req.file.path) fs.unlinkSync(req.file.path);
        return res.status(401).json({ error: 'Token Invalid' });
    }
    if (!number || !req.file) {
        if (req.file && req.file.path) fs.unlinkSync(req.file.path);
        return res.status(400).json({ error: 'number and file are required' });
    }
    if (serverStatus !== 'waReady') {
        if (req.file && req.file.path) fs.unlinkSync(req.file.path);
        return res.status(400).json({ error: 'WhatsApp belum siap' });
    }

    const filePath = req.file.path;
    try {
        const media = MessageMedia.fromFilePath(filePath);
        const chatId = formatNumberToWhatsAppId(number);

        const chat = await client.getChatById(chatId);

        const options = {
            sendSeen: false // 🔥 WAJIB
        };

        if (!asDocument && caption) options.caption = caption;
        if (asDocument) options.sendMediaAsDocument = true;

        const sent = await chat.sendMessage(media, options);

        fs.unlinkSync(filePath);
        return res.json({ status: 'ok', id: sent.id._serialized });

    } catch (err) {
        console.error(err);
        if (req.file && req.file.path && fs.existsSync(req.file.path)) {
            fs.unlinkSync(req.file.path);
        }
        return res.status(500).json({ error: err.message });
    }
});

// 4️⃣ LOGOUT & RESET (QR BARU)
app.get('/logout', async (req, res) => {
    if (req.query.token !== myToken) {
        return res.status(401).json({ error: 'Token Invalid' });
    }

    try {
        await client.logout();
        serverStatus = 'qrcode';
        qrCode = '';

        await client.initialize();

        res.json({ status: 'logout & restart' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// ====================== SERVER ======================
app.listen(PORT, () => {
    serverStatus = 'portReady';
    console.log(`Server berjalan di port ${PORT}`);

    // START OUTBOX RANDOM EXECUTOR
    const firstDelay = getRandomDelay();
    console.log(`⏳ First Outbox execution in ${firstDelay / 1000}s\n`);

    setTimeout(executeOutboxProcess, firstDelay);
});


function getRandomDelay() {
    return Math.floor(Math.random() * (MAX_DELAY - MIN_DELAY + 1)) + MIN_DELAY;
}


async function executeOutboxProcess() {
    try {
        console.log('🔄 Execute Outbox:', new Date().toISOString());

        const response = await fetch(OUTBOX_URL);
        const text = await response.text();

        console.log('✅ Outbox response:', text);

    } catch (err) {
        console.error('❌ Outbox error:', err.message);
    } finally {
        const nextDelay = getRandomDelay();
        console.log(`⏳ Next execution in ${nextDelay / 1000}s\n`);

        setTimeout(executeOutboxProcess, nextDelay);
    }
}
